/***************************************************************************
 *   Copyright (C) 2000 by Torsten Henschel                                *
 *   thenschel@henschelsoft.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <malloc.h>
#include <stdio.h>
#include <unistd.h>

#include <make_popup.h>
#include <misc.h>

#include <kaction.h>
#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>

#include <qfile.h>
#include <qlabel.h>
#include <qlayout.h>

/*
 * contructor of Make_Popup
 */
Make_Popup::Make_Popup(QWidget *parent, const char *name, QString sender) : KDialog(parent, name, true) {

    QGridLayout *layout = new QGridLayout(this);

    QLabel *lbl_Group = new QLabel(i18n("Group:"), this);
    lbl_Group -> setMinimumWidth(50);
    layout -> addWidget(lbl_Group, 1, 1);

    cb_Group = new KComboBox(true, this);
    cb_Group -> setMinimumWidth(250);
    layout -> addWidget(cb_Group, 1, 3);
    cb_Group -> setEditable(false);
    connect(cb_Group, SIGNAL(activated(const QString &)), SLOT(changeGroupBox(const QString &)));

    QLabel *lbl_host = new QLabel(i18n("Host:"), this);
    lbl_host -> setMinimumWidth(50);
    layout -> addWidget(lbl_host, 3, 1);

    cb_Host = new KComboBox(true, this);
    cb_Host -> setMinimumWidth(250);
    layout -> addWidget(cb_Host, 3, 3);

    QLabel *lbl_Text = new QLabel(i18n("Text:"), this);
    layout -> addWidget(lbl_Text, 5, 1);

    mle_Text = new QMultiLineEdit(this);
    mle_Text -> setMinimumHeight(180);
    mle_Text -> setMinimumWidth(350);
    layout -> addMultiCellWidget(mle_Text, 7, 7, 1, 3);

    KPushButton *btn_Ok = new KPushButton(i18n("Ok"), this);
    layout -> addWidget(btn_Ok, 9, 1);
    connect(btn_Ok, SIGNAL(clicked()), SLOT(sendPopup()));

    KPushButton *btn_Cancel = new KPushButton(i18n("Cancel"), this );
    layout -> addWidget(btn_Cancel, 9, 3);
    connect(btn_Cancel, SIGNAL(clicked()), SLOT(reject()));

    layout -> addColSpacing(0, 10);
    layout -> addColSpacing(2, 5);
    layout -> addColSpacing(4, 10);

    layout -> addRowSpacing(0, 10);
    layout -> addRowSpacing(2, 5);
    layout -> addRowSpacing(4, 5);
    layout -> addRowSpacing(6, 5);
    layout -> addRowSpacing(8, 5);
    layout -> addRowSpacing(10, 10);

    // hot keys
    KActionCollection *actColl = new KActionCollection(this);
    new KAction("send", CTRL+Key_S, this, SLOT(sendPopup()), actColl, "send");
    new KAction("cancel", CTRL+Key_C, this, SLOT(reject()), actColl, "cancel");

    if (sender.isEmpty()) {
        cb_Group -> insertItem("NO GROUP", -1);
        readGroupList();
    } else {
        cb_Group -> setEnabled(false);
        cb_Host -> insertItem(sender);
        cb_Host -> setEnabled(false);
    }
}


/*
 * destructor of Make_Popup
 */
Make_Popup::~Make_Popup() {
}


/*
 * method sendPopup
 */
void Make_Popup::sendPopup() {

    int linecount = 0;

    QString cmd;
    cmd.append(SMBPATH);
    cmd.append("/smbclient -M '");
    cmd.append(cb_Host -> currentText());
    cmd.append("' -N -U '");
    cmd.append(getHostName());
    cmd.append("'");

    kdDebug() << cmd.local8Bit() << endl;

    // open process
    FILE *sendprocess = popen(cmd.local8Bit(), "w");

    // write data
    if (sendprocess != NULL) {
        while (linecount < mle_Text -> numLines()) {
            fprintf(sendprocess, "%s\n", (const char *)mle_Text -> textLine(linecount).utf8());
            linecount++;
        }
        if (pclose(sendprocess) == 0) {
            KMessageBox::information(this, i18n("Popup sent!"), i18n("Success"));
            reject();
        } else {
            KMessageBox::sorry(this, i18n("Popup not sent!"), i18n("Error"));
        }
    }

}


/*
 * readHostName
 */
QString Make_Popup::getHostName() {

    // todo read name from samba
    char *tmp = (char *) malloc (255 * (sizeof(char)));
    gethostname(tmp, 255);
    QString hostname = tmp;
    hostname = hostname.upper();
    if (hostname.contains('.') != 0) {
        hostname.remove(hostname.find('.'), hostname.length());
    }
    return hostname;
}


/*
 * read available groups from samba (localhost)
 */
void Make_Popup::readGroupList() {

    QString cmd;
    cmd.append(SMBPATH);
    cmd.append("/smbclient -L ");
    cmd.append("localhost");
    cmd.append(" -N");

    // open process
    FILE *sambaQuery = popen(cmd.local8Bit(), "r");

    // read data
    if (sambaQuery != NULL) {
        QTextStream stream (sambaQuery, IO_ReadOnly);
        QString line;
        bool groupLine = false;
        bool sepLine = false;
        while (! stream.eof()) {
            line = stream.readLine();
            if (! groupLine && line.contains("\tWorkgroup            Master", true) == 1) {
                groupLine = true;
            } else if (groupLine && ! sepLine && line.contains("\t---------            -------", true) == 1) {
                sepLine = true;
            } else if (sepLine) {
                if (line.length() == 0) {
                    break;
                }
                line.remove(0, 1);
                QString group = line.mid(0, line.find("  ", 0, true));
                kdDebug() << "found group = '" << group << "'" << endl;
                cb_Group -> insertItem(group, -1);
            }
        }
    }
    else {
        QString errorMsg;
        errorMsg.append("can't open process='read group list' ");
        errorMsg.append(cmd);
        perror(errorMsg);
    }

    // close process
    if (sambaQuery != NULL) {
        if (pclose(sambaQuery) != 0) {
            QString errorMsg;
            errorMsg.append("can't close process='read group list' ");
            errorMsg.append(cmd);
            perror(errorMsg);
        }
    }
}


/*
 * read available host from group with samba
 */
void Make_Popup::readHostList(QString group) {

    cb_Host -> clear();
    cb_Host -> insertItem("NO HOST", -1);
    if (group == "NO GROUP") {
        return;
    }

    QString host;
    QString cmd;

    // search master of group
    cmd.append(SMBPATH);
    cmd.append("/smbclient -L ");
    cmd.append("localhost");
    cmd.append(" -N");

    // open process
    FILE *sambaQuery = popen(cmd.local8Bit(), "r");

    // read data
    if (sambaQuery != NULL) {
        QTextStream stream(sambaQuery, IO_ReadOnly);
        QString line;
        int groupLine = 0;
        int sepLine = 0;
        while (! stream.eof()) {
            line = stream.readLine();
            if (groupLine == 0 && line.contains("\tWorkgroup            Master", true) == 1) {
                groupLine = 1;
            } else if (groupLine == 1 && sepLine != 1 && line.contains("\t---------            -------", true) == 1) {
                sepLine = 1;
            } else if (sepLine == 1) {
                if (line.length() == 0) {
                    break;
                }
                line.remove(0, 1);
                QString tmp = line.mid(0, line.find("  ", 0, true));
                kdDebug() << "found group = '" << tmp << "'" << endl;
                if (tmp == group) {
                    line.remove(0, line.find("  ", 0, true));
                    host = line.stripWhiteSpace();
                    kdDebug() << "found master host= '" << host << "' in group = '" << tmp << "'" << endl;
                    break;
                }
            }
        }
    }
    else {
        QString errorMsg;
        errorMsg.append("can't open process='read master of group' ");
        errorMsg.append(cmd);
        perror(errorMsg);
    }

    // close process
    if (sambaQuery != NULL) {
        if (pclose(sambaQuery) != 0) {
            QString errorMsg;
            errorMsg.append("can't close process='read master of group' ");
            errorMsg.append(cmd);
            perror(errorMsg);
        }
    }

    // no master fround for group
    if (host.isEmpty()) {
        return;
    }

    // search hosts is group
    cmd.truncate(0);
    cmd.append(SMBPATH);
    cmd.append("/smbclient -L '");
    cmd.append(host);
    cmd.append("' -N");

    // open process
    sambaQuery = popen(cmd.local8Bit(), "r");

    // read data
    if (sambaQuery != NULL) {
        QTextStream stream(sambaQuery, IO_ReadOnly);
        QString line;
        int groupLine = 0;
        int sepLine = 0;
        while (! stream.eof()) {
            line = stream.readLine();
            if (groupLine == 0 && line.contains("\tServer               Comment", true) == 1) {
                groupLine = 1;
            } else if (groupLine == 1 && sepLine != 1 && line.contains("\t---------            -------", true) == 1) {
                sepLine = 1;
            } else if (sepLine == 1) {
                if (line.length() == 0) {
                    break;
                }
                line.remove(0, 1);
                host = line.mid(0, line.find("  ", 0, true));
                if (! host.isEmpty()) {
                    kdDebug() << "found host= '" << host << "' in group = '" << group << "'" << endl;
                    cb_Host -> insertItem(host, -1);
                }
            }
        }
    }
    else {
        QString errorMsg;
        errorMsg.append("can't open process='read hosts from master' ");
        errorMsg.append(cmd);
        perror(errorMsg);
    }

    // close process
    if (sambaQuery != NULL) {
        if (pclose(sambaQuery) != 0) {
            QString errorMsg;
            errorMsg.append("can't close process='read hosts from master' ");
            errorMsg.append(cmd);
            perror(errorMsg);
        }
    }
}


/*
 * will be called, if groupbox is changed
 */
void Make_Popup::changeGroupBox(const QString &group) {

    readHostList(group);
}
