/***************************************************************************
 kpopup.cpp - create mainwindow

 copyright : (C) 2000 by Torsten Henschel
 email     : thenschel@henschelsoft.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>

#include <kiconloader.h>
#include <klocale.h>
#include <kstdaction.h>
#include <ktoolbarbutton.h>

#include <qlayout.h>
#include <qtimer.h>
#include <qtooltip.h>

#include "kpopup.h"
#include "make_popup.h"
#include "misc.h"
#include "resource.h"

/*
 * constructor of KPopup
 */
KPopup::KPopup(QWidget *, const char *name) : KMainWindow (0L, name) {

    actual = 0;

    QWidget *mainWidget = new QWidget(this, "mainWidget");
    setCentralWidget(mainWidget);
    QGridLayout *layout = new QGridLayout(mainWidget);
       
    init();
    initToolbar();
    initStatusbar();

    toolbar -> show();
    statusbar -> show();
    addToolBar(toolbar);

    setStatusText(i18n("Welcome to KPopup ") + QString(VERSION));
    timer -> start(3000, TRUE);

    tray = new KSystemTray(this);
    tray -> setPixmap(kapp -> miniIcon());
    tray -> show();

    QLabel *lbl_Sender = new QLabel(i18n("Popup from:"), mainWidget);
    lbl_Sender -> setMinimumWidth(100);
    lbl_Sender -> setAlignment(Qt::AlignLeft);
    layout -> addWidget(lbl_Sender, 1, 1);

    lbl_SenderValue = new QLabel("", mainWidget);
    lbl_SenderValue -> setFrameStyle(QFrame::Panel | QFrame::Sunken);
    lbl_SenderValue -> setAlignment(Qt::AlignLeft);
    lbl_SenderValue -> setMinimumWidth(250);
    layout -> addWidget(lbl_SenderValue, 1, 2);
    
    QLabel *lbl_DateTime = new QLabel(i18n("Popup time:"), mainWidget);
    lbl_DateTime -> setMinimumWidth(100);
    lbl_DateTime -> setAlignment(Qt::AlignLeft);
    layout -> addWidget(lbl_DateTime, 3, 1);
    
    lbl_DateTimeValue = new QLabel("", mainWidget);
    lbl_DateTimeValue -> setFrameStyle(QFrame::Panel | QFrame::Sunken);
    lbl_DateTimeValue -> setAlignment(Qt::AlignLeft);
    lbl_DateTimeValue -> setMinimumWidth(250);
    layout -> addWidget(lbl_DateTimeValue, 3, 2);
    
    tv_Text = new QTextView(mainWidget);
    layout -> addMultiCellWidget(tv_Text, 5, 5, 1, 2);

    layout -> addColSpacing(0, 10);
    layout -> addColSpacing(3, 10);

    layout -> addRowSpacing(0, 10);
    layout -> addRowSpacing(2, 5);
    layout -> addRowSpacing(4, 5);
    layout -> addRowSpacing(6, 10);

    // hot keys
    KStdAction::quit(this, SLOT(exitKPopup()), actionCollection());
    new KAction("new", CTRL+Key_N, this, SLOT(newPopup()), actionCollection(), "new");
    actReply = new KAction("reply", CTRL+Key_R, this, SLOT(answerPopup()), actionCollection(), "reply");
    actNext = new KAction("next", ALT+Key_N, this, SLOT(nextPopup()), actionCollection(), "next");
    actPrev = new KAction("previous", ALT+Key_P, this, SLOT(previousPopup()), actionCollection(), "previous");
    actDel = new KAction("delete", ALT+Key_D, this, SLOT(deletePopup()), actionCollection(), "delete");
    KStdAction::help(this, SLOT(helpKPopup()), actionCollection());

    // set Button in toolbar
    checkMap();
}


/*
 * destructor of KPopup
 */
KPopup::~KPopup() {
}


/*
 * method init
 */
void KPopup::init() {

    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(timerDone()));
}


/*
 * method initialize the toolbar
 */
void KPopup::initToolbar() {

    toolbar = new KToolBar(this);

    toolbar -> insertButton("exit", ID_EXIT, true, i18n("Quit"));
    toolbar -> insertSeparator();
    toolbar -> insertButton("filenew", ID_NEW_POPUP, true, i18n("New"));
    toolbar -> insertButton("reload", ID_ANSWER_POPUP, true, i18n("Reply"));
    toolbar -> insertButton("back", ID_PREVIOUS_POPUP, true, i18n("Previous"));
    toolbar -> insertButton("forward", ID_NEXT_POPUP, true, i18n("Next"));
    toolbar -> insertButton("remove", ID_DELETE_POPUP, true, i18n("Delete"));
    toolbar -> insertSeparator();
    toolbar -> insertSeparator();
    toolbar -> insertButton("help", ID_HELP, true, i18n("Help"));
    
    connect(toolbar, SIGNAL(clicked(int)), SLOT(commandCallback(int)));
}


/*
 * method initialisize the statusbar
 */
void KPopup::initStatusbar() {

    statusbar = new KStatusBar(this);
    statusBar() -> insertItem("", ID_STATUS_TEXT, 1, true);
}


/*
 * method handles events
 */
void KPopup::commandCallback(int id) {

    switch(id) {
        case ID_EXIT:
                exitKPopup();
                break;
        case ID_NEW_POPUP:
                newPopup();
                break;
        case ID_ANSWER_POPUP:
                answerPopup();
                break;
        case ID_HELP:
                helpKPopup();
                break;
        case ID_PREVIOUS_POPUP:
                previousPopup();
                break;
        case ID_NEXT_POPUP:
                nextPopup();
                break;
        case ID_DELETE_POPUP:
                deletePopup();
                break;
    }
}


/*
 * method timerDone
 */
void KPopup::timerDone() {

    QString number;

    QString statusText;
    statusText.append("Popup ");
    number.setNum(actual, 10);
    statusText.append(number);
    statusText.append("/");
    number.setNum(senderMap.count(), 10);
    statusText.append(number);

    setStatusText(statusText);

    QString trayText;
    trayText.append("KPopup - ");
    number.setNum(actual, 10);
    trayText.append(number);
    trayText.append("/");
    number.setNum(senderMap.count(), 10);
    trayText.append(number);
    
    QToolTip::remove(tray);
    QToolTip::add(tray, trayText);
}


/*
 * method setStattusText
 */
void KPopup::setStatusText(QString text) {

    statusbar -> changeItem(text, ID_STATUS_TEXT);
}


/*
 * method newPopup
 */
void KPopup::newPopup() {

    Make_Popup *newPopupView = new Make_Popup(this, "New Popup", "");
    newPopupView -> setCaption(i18n("New Popup"));
    newPopupView -> exec();
}


/*
 * method answerPopup
 */
void KPopup::answerPopup() {

    QString sender = lbl_SenderValue -> text();

    Make_Popup *answerPopupView = new Make_Popup(this, "Reply Popup", sender);
    answerPopupView -> setCaption(i18n("Reply Popup"));
    answerPopupView -> exec();
}


/*
 * method signalNewMessage
 */
void KPopup::signalNewMessage() {

#ifdef DEBUG
    fprintf(stderr, "signal received\n");
#endif

    actual = senderMap.count() + 1;
    senderMap.insert(actual, readSender());
    popupTextMap.insert(actual, readPopupText());
    currentTime = new QTime();
    *currentTime = currentTime -> currentTime();
    currentDate = new QDate();
    *currentDate = currentDate -> currentDate();
    timeMap.insert(actual, currentDate -> toString() + " " + currentTime -> toString());

    showPopup();
    kapp -> beep();
    showMaximized();
}


/*
 * method checked maps and set buttons in toolbar
 */
void KPopup::checkMap() {
    int count = senderMap.count();

    if (count != 0) {
        toolBar() -> getButton(ID_DELETE_POPUP) -> setEnabled(true);
        actDel -> setEnabled(true);
        toolBar() -> getButton(ID_ANSWER_POPUP) -> setEnabled(true);
        actReply -> setEnabled(true);
    } else {
        toolBar() -> getButton(ID_DELETE_POPUP) -> setEnabled(false);
        actDel -> setEnabled(false);
        toolBar() -> getButton(ID_ANSWER_POPUP) -> setEnabled(false);
        actReply -> setEnabled(false);
    }

    if (count > 1 && actual > 1) {
        toolBar() -> getButton(ID_PREVIOUS_POPUP) -> setEnabled(true);
        actPrev -> setEnabled(true);
    } else {
        toolBar() -> getButton(ID_PREVIOUS_POPUP) -> setEnabled(false);
        actPrev -> setEnabled(false);
    }

    if (count > 1 && actual < count) {
        toolBar() -> getButton(ID_NEXT_POPUP) -> setEnabled(true);
        actNext -> setEnabled(true);
    } else {
        toolBar() -> getButton(ID_NEXT_POPUP) -> setEnabled(false);
        actNext -> setEnabled(false);
    }
}


/*
 * method displayed popup
 */
void KPopup::showPopup() {

    if (!senderMap.isEmpty()) {
        lbl_SenderValue -> setText(senderMap[actual]);
        lbl_DateTimeValue -> setText(timeMap[actual]);
        tv_Text -> setText(popupTextMap[actual]);
    } else {
        lbl_SenderValue -> setText("");
        lbl_DateTimeValue -> setText("");
        tv_Text -> setText("");
    }

    checkMap();
    timerDone();
}


/*
 * method remove a popup from maps
 */
void KPopup::deletePopup() {
    int count = senderMap.count();

    if (actual == count) {
        senderMap.remove(actual);
        timeMap.remove(actual);
        popupTextMap.remove(actual);
        actual--;
    } else {
        for (int i = actual; i <= count - 1; i++) {
            senderMap.insert(i, senderMap[i + 1]);
            senderMap.remove(i + 1);
            timeMap.insert(i, timeMap[i + 1]);
            timeMap.remove(i + 1);
            popupTextMap.insert(i, popupTextMap[i + 1]);
            popupTextMap.remove(i + 1);
        }
    }

    showPopup();
}

/*
 * hide event from KMainWindow
 */
void KPopup::hideEvent(QHideEvent *e) {

    KMainWindow::hideEvent(e);
    tray -> show();
    hide();
}

/*
 * exit Kpopup
 */
void KPopup::exitKPopup() {

    kapp -> quit();
}


/*
 * help
 */
void KPopup::helpKPopup() {

    kapp -> invokeHelp("", "");
}


/*
 * next Popup
 */
void KPopup::nextPopup() {

    actual++;
    showPopup();
}


/*
 * previous Popup
 */
void KPopup::previousPopup() {

    actual--;
    showPopup();
}
